
    // ===== Dynamic Header: Clock =====
    const clock = document.querySelector('#clock span');
    function pad(n) { return (n < 10 ? '0' : '') + n }
    function tick() {
      const d = new Date();
      const h = pad(d.getHours());
      const m = pad(d.getMinutes());
      clock.textContent = `${h}:${m}`;
    }
    setInterval(tick, 1000); tick();

    // ===== Sidebar: dropdowns =====
    const dropdownTriggers = [...document.querySelectorAll('[data-toggle="submenu"]')];
    dropdownTriggers.forEach(btn => {
      const item = btn.closest('.nav-item');
      const pane = item.querySelector('.submenu');
      btn.addEventListener('click', () => {
        const expanded = btn.getAttribute('aria-expanded') === 'true';
        btn.setAttribute('aria-expanded', String(!expanded));
        btn.classList.toggle('active', !expanded);
        pane.style.height = !expanded ? (pane.scrollHeight + 'px') : '0px';
        btn.querySelector('.chev').style.transform = !expanded ? 'rotate(180deg)' : 'rotate(0)';
      });
    });
    function closeSiblings(currentBtn) {
      dropdownTriggers.forEach(btn => {
        if (btn !== currentBtn) {
          const pane = btn.closest('.nav-item').querySelector('.submenu');
          btn.setAttribute('aria-expanded', 'false');
          btn.classList.remove('active');
          pane.style.height = '0px';
          btn.querySelector('.chev').style.transform = 'rotate(0)';
        }
      })
    }
    dropdownTriggers.forEach(btn => btn.addEventListener('click', () => closeSiblings(btn)));

    // ===== Sidebar: collapse & mobile toggle =====
    const app = document.getElementById('app');
    const sidebar = document.getElementById('sidebar');
    const sidebarToggle = document.getElementById('sidebarToggle');
    const collapseToggle = document.getElementById('collapseToggle');
    const closeSidebarMobile = document.getElementById('closeSidebarMobile');
    
    sidebarToggle.addEventListener('click', () => {
      if (window.matchMedia('(max-width: 860px)').matches) {
        sidebar.classList.toggle('open');
      } else {
        app.classList.toggle('collapsed');
      }
    });
    
    collapseToggle.addEventListener('click', () => {
      app.classList.toggle('collapsed');
    });
    
    closeSidebarMobile.addEventListener('click', () => {
      sidebar.classList.remove('open');
    });
    
    document.querySelector('.main').addEventListener('click', () => {
      if (window.matchMedia('(max-width: 860px)').matches) {
        sidebar.classList.remove('open');
      }
    });

    // ===== Keyboard shortcut for search focus =====
    const searchInput = document.querySelector('.header .search input');
    window.addEventListener('keydown', (e) => {
      if (e.key === '/' && document.activeElement !== searchInput) {
        e.preventDefault(); searchInput.focus();
      }
    });

    // ===== Set current year in footer =====
    document.getElementById('year').textContent = new Date().getFullYear();

    // ===== Sample Pending Data =====
    const pendingData = [
      {
        id: 'PEND-2024-001',
        customerId: 'CUS-001',
        name: 'Ahmed Raza',
        company: 'Tech Solutions Ltd',
        itemType: 'Subscription Renewal',
        createdDate: '2024-04-10',
        dueDate: '2024-04-25',
        priority: 'high',
        status: 'pending',
        email: 'ahmed@techsolutions.com',
        daysOverdue: 0
      },
      {
        id: 'PEND-2024-002',
        customerId: 'CUS-002',
        name: 'Sana Khan',
        company: 'Digital Marketing Pro',
        itemType: 'Payment Approval',
        createdDate: '2024-04-05',
        dueDate: '2024-04-15',
        priority: 'high',
        status: 'overdue',
        email: 'sana@dmp.com',
        daysOverdue: 5
      },
      {
        id: 'PEND-2024-003',
        customerId: 'CUS-003',
        name: 'Bilal Ahmed',
        company: 'Startup Innovations',
        itemType: 'Document Verification',
        createdDate: '2024-04-12',
        dueDate: '2024-04-30',
        priority: 'medium',
        status: 'in-review',
        email: 'bilal@startup.com',
        daysOverdue: 0
      },
      {
        id: 'PEND-2024-004',
        customerId: 'CUS-004',
        name: 'Hira Malik',
        company: 'E-commerce Store',
        itemType: 'Support Ticket',
        createdDate: '2024-04-08',
        dueDate: '2024-04-22',
        priority: 'medium',
        status: 'pending',
        email: 'hira@estore.com',
        daysOverdue: 0
      },
      {
        id: 'PEND-2024-005',
        customerId: 'CUS-005',
        name: 'Usman Ali',
        company: 'Consulting Firm',
        itemType: 'Account Approval',
        createdDate: '2024-04-01',
        dueDate: '2024-04-10',
        priority: 'high',
        status: 'overdue',
        email: 'usman@consulting.com',
        daysOverdue: 10
      },
      {
        id: 'PEND-2024-006',
        customerId: 'CUS-006',
        name: 'Fatima Noor',
        company: 'Creative Agency',
        itemType: 'Subscription Renewal',
        createdDate: '2024-04-14',
        dueDate: '2024-04-28',
        priority: 'medium',
        status: 'awaiting-approval',
        email: 'fatima@creative.com',
        daysOverdue: 0
      },
      {
        id: 'PEND-2024-007',
        customerId: 'CUS-007',
        name: 'Kamran Butt',
        company: 'Manufacturing Corp',
        itemType: 'Payment Approval',
        createdDate: '2024-04-03',
        dueDate: '2024-04-18',
        priority: 'low',
        status: 'pending',
        email: 'kamran@manufacturing.com',
        daysOverdue: 0
      },
      {
        id: 'PEND-2024-008',
        customerId: 'CUS-008',
        name: 'Ayesha Siddiqui',
        company: 'Healthcare Solutions',
        itemType: 'Document Verification',
        createdDate: '2024-04-09',
        dueDate: '2024-04-24',
        priority: 'medium',
        status: 'in-review',
        email: 'ayesha@healthcare.com',
        daysOverdue: 0
      },
      {
        id: 'PEND-2024-009',
        customerId: 'CUS-009',
        name: 'Zain Abid',
        company: 'Real Estate Partners',
        itemType: 'Support Ticket',
        createdDate: '2024-04-06',
        dueDate: '2024-04-16',
        priority: 'high',
        status: 'overdue',
        email: 'zain@realestate.com',
        daysOverdue: 4
      },
      {
        id: 'PEND-2024-010',
        customerId: 'CUS-010',
        name: 'Nadia Shah',
        company: 'Education First',
        itemType: 'Account Approval',
        createdDate: '2024-04-13',
        dueDate: '2024-04-27',
        priority: 'medium',
        status: 'awaiting-approval',
        email: 'nadia@education.com',
        daysOverdue: 0
      }
    ];

    // ===== Populate Pending Table =====
    function populatePendingTable(filteredData = pendingData) {
      const tableBody = document.getElementById('pendingTable');
      tableBody.innerHTML = '';
      
      filteredData.forEach(item => {
        const row = document.createElement('tr');
        
        // Get initials for avatar
        const initials = item.name.split(' ').map(n => n[0]).join('').toUpperCase();
        
        // Format dates
        const createdDate = new Date(item.createdDate).toLocaleDateString('en-US', {
          year: 'numeric',
          month: 'short',
          day: 'numeric'
        });
        
        const dueDate = new Date(item.dueDate).toLocaleDateString('en-US', {
          year: 'numeric',
          month: 'short',
          day: 'numeric'
        });
        
        // Determine days overdue class
        let daysClass = 'normal';
        if (item.daysOverdue > 7) {
          daysClass = 'critical';
        } else if (item.daysOverdue > 0) {
          daysClass = 'warning';
        }
        
        row.innerHTML = `
          <td>
            <div style="font-weight: 600; color: var(--primary);">${item.id}</div>
          </td>
          <td>
            <div style="display: flex; align-items: center; gap: 12px;">
              <div class="customer-avatar">${initials}</div>
              <div>
                <div style="font-weight: 600;">${item.name}</div>
                <div style="font-size: 12px; color: var(--muted);">${item.email}</div>
              </div>
            </div>
          </td>
          <td>${item.itemType}</td>
          <td>${createdDate}</td>
          <td>
            <div>${dueDate}</div>
            ${item.daysOverdue > 0 ? `
              <div class="days-overdue ${daysClass}" style="margin-top: 4px;">
                ${item.daysOverdue} days overdue
              </div>
            ` : ''}
          </td>
          <td>
            <div style="display: flex; align-items: center; gap: 6px;">
              <div style="width: 8px; height: 8px; border-radius: 50%; background: ${
                item.priority === 'high' ? 'var(--danger)' : 
                item.priority === 'medium' ? 'var(--warning)' : 
                'var(--success)'
              };"></div>
              <span style="text-transform: capitalize;">${item.priority}</span>
            </div>
          </td>
          <td>
            <span class="status ${item.status}">${item.status.charAt(0).toUpperCase() + item.status.slice(1).replace('-', ' ')}</span>
          </td>
          <td>
            <div class="action-buttons">
              <button class="action-btn view" title="View Details" data-id="${item.id}">
                <i class="fa-solid fa-eye"></i>
              </button>
              <button class="action-btn approve" title="Approve" data-id="${item.id}">
                <i class="fa-solid fa-check"></i>
              </button>
              ${item.status === 'overdue' || item.status === 'pending' ? `
                <button class="action-btn remind" title="Send Reminder" data-id="${item.id}">
                  <i class="fa-solid fa-bell"></i>
                </button>
              ` : ''}
              <button class="action-btn reject" title="Reject" data-id="${item.id}">
                <i class="fa-solid fa-times"></i>
              </button>
            </div>
          </td>
        `;
        
        tableBody.appendChild(row);
      });
    }

    // ===== Table Search Functionality =====
    document.getElementById('tableSearch').addEventListener('input', function(e) {
      const searchTerm = e.target.value.toLowerCase();
      const rows = document.querySelectorAll('#pendingTable tr');
      
      rows.forEach(row => {
        const text = row.textContent.toLowerCase();
        row.style.display = text.includes(searchTerm) ? '' : 'none';
      });
    });

    // ===== Filter Functionality =====
    document.getElementById('applyFilters').addEventListener('click', function() {
      const typeFilter = document.getElementById('typeFilter').value;
      const statusFilter = document.getElementById('statusFilter').value;
      const priorityFilter = document.getElementById('priorityFilter').value;
      
      let filteredData = pendingData;
      
      // Apply type filter
      if (typeFilter) {
        filteredData = filteredData.filter(item => 
          item.itemType.toLowerCase().includes(typeFilter.toLowerCase())
        );
      }
      
      // Apply status filter
      if (statusFilter) {
        filteredData = filteredData.filter(item => item.status === statusFilter);
      }
      
      // Apply priority filter
      if (priorityFilter) {
        filteredData = filteredData.filter(item => item.priority === priorityFilter);
      }
      
      populatePendingTable(filteredData);
      document.querySelector('.pagination-info').textContent = `Showing 1-${filteredData.length} of ${filteredData.length} pending items`;
    });

    // ===== Clear Filters =====
    document.getElementById('clearFilters').addEventListener('click', function() {
      document.getElementById('typeFilter').value = '';
      document.getElementById('statusFilter').value = '';
      document.getElementById('priorityFilter').value = '';
      document.getElementById('tableSearch').value = '';
      populatePendingTable();
      document.querySelector('.pagination-info').textContent = 'Showing 1-10 of 142 pending items';
    });

    // ===== Export Functionality =====
    document.getElementById('exportReport').addEventListener('click', function() {
      alert('Exporting pending report...');
      // In a real application, this would generate and download a CSV/PDF report
    });

    // ===== Print Functionality =====
    document.getElementById('printReport').addEventListener('click', function() {
      alert('Opening print dialog...');
      // In a real application, this would open the browser's print dialog
    });

    // ===== Action Button Handlers =====
    document.addEventListener('click', function(e) {
      if (e.target.closest('.action-btn.view')) {
        const itemId = e.target.closest('.action-btn').getAttribute('data-id');
        alert(`View pending item details for ID: ${itemId}`);
      }
      
      if (e.target.closest('.action-btn.approve')) {
        const itemId = e.target.closest('.action-btn').getAttribute('data-id');
        if (confirm('Are you sure you want to approve this item?')) {
          alert(`Item approved for ID: ${itemId}`);
          // In a real application, you would update the item status
        }
      }
      
      if (e.target.closest('.action-btn.reject')) {
        const itemId = e.target.closest('.action-btn').getAttribute('data-id');
        if (confirm('Are you sure you want to reject this item?')) {
          alert(`Item rejected for ID: ${itemId}`);
          // In a real application, you would update the item status
        }
      }
      
      if (e.target.closest('.action-btn.remind')) {
        const itemId = e.target.closest('.action-btn').getAttribute('data-id');
        alert(`Reminder sent for pending item ID: ${itemId}`);
      }
    });

    // ===== Pagination Controls =====
    document.querySelectorAll('.page-btn').forEach(btn => {
      btn.addEventListener('click', function() {
        if (!this.querySelector('i')) { // Don't change active state for arrow buttons
          document.querySelectorAll('.page-btn').forEach(b => b.classList.remove('active'));
          this.classList.add('active');
        }
      });
    });

    // ===== Initialize the page =====
    document.addEventListener('DOMContentLoaded', function() {
      // Set current page as active in sidebar
      const currentPage = 'pendingreport.html';
      document.querySelectorAll('.nav a').forEach(link => {
        if (link.getAttribute('href') === currentPage) {
          link.classList.add('active');
          // If it's in a submenu, open the parent dropdown
          const parentItem = link.closest('.nav-item');
          if (parentItem) {
            const dropdownBtn = parentItem.querySelector('[data-toggle="submenu"]');
            if (dropdownBtn) {
              dropdownBtn.click(); // Open the dropdown
            }
          }
        }
      });
      
      // Populate the pending table
      populatePendingTable();
    });
 