
    // ===== Dynamic Header: Clock =====
    const clock = document.querySelector('#clock span');
    function pad(n) { return (n < 10 ? '0' : '') + n }
    function tick() {
      const d = new Date();
      const h = pad(d.getHours());
      const m = pad(d.getMinutes());
      clock.textContent = `${h}:${m}`;
    }
    setInterval(tick, 1000); tick();

    // ===== Sidebar: dropdowns =====
    const dropdownTriggers = [...document.querySelectorAll('[data-toggle="submenu"]')];
    dropdownTriggers.forEach(btn => {
      const item = btn.closest('.nav-item');
      const pane = item.querySelector('.submenu');
      btn.addEventListener('click', () => {
        const expanded = btn.getAttribute('aria-expanded') === 'true';
        btn.setAttribute('aria-expanded', String(!expanded));
        btn.classList.toggle('active', !expanded);
        pane.style.height = !expanded ? (pane.scrollHeight + 'px') : '0px';
        btn.querySelector('.chev').style.transform = !expanded ? 'rotate(180deg)' : 'rotate(0)';
      });
    });
    function closeSiblings(currentBtn) {
      dropdownTriggers.forEach(btn => {
        if (btn !== currentBtn) {
          const pane = btn.closest('.nav-item').querySelector('.submenu');
          btn.setAttribute('aria-expanded', 'false');
          btn.classList.remove('active');
          pane.style.height = '0px';
          btn.querySelector('.chev').style.transform = 'rotate(0)';
        }
      })
    }
    dropdownTriggers.forEach(btn => btn.addEventListener('click', () => closeSiblings(btn)));

    // ===== Sidebar: collapse & mobile toggle =====
    const app = document.getElementById('app');
    const sidebar = document.getElementById('sidebar');
    const sidebarToggle = document.getElementById('sidebarToggle');
    const collapseToggle = document.getElementById('collapseToggle');
    const closeSidebarMobile = document.getElementById('closeSidebarMobile');
    sidebarToggle.addEventListener('click', () => {
      if (window.matchMedia('(max-width: 860px)').matches) {
        sidebar.classList.toggle('open');
      } else {
        app.classList.toggle('collapsed');
      }
    });
    collapseToggle.addEventListener('click', () => {
      app.classList.toggle('collapsed');
    });
    closeSidebarMobile.addEventListener('click', () => {
      sidebar.classList.remove('open');
    });
    document.getElementById('main').addEventListener('click', () => {
      if (window.matchMedia('(max-width: 860px)').matches) {
        sidebar.classList.remove('open');
      }
    });

    // ===== Keyboard shortcut for search focus =====
    const searchInput = document.querySelector('.header .search input');
    window.addEventListener('keydown', (e) => {
      if (e.key === '/' && document.activeElement !== searchInput) {
        e.preventDefault(); searchInput.focus();
      }
    });

    // ===== Set current year in footer =====
    document.getElementById('year').textContent = new Date().getFullYear();

    // ===== Sample Customer Data =====
    const customers = [
      {
        id: 'CUS-001',
        name: 'Ahmed Raza',
        company: 'Tech Solutions Ltd',
        phone: '+92 300 1234567',
        email: 'ahmed@techsolutions.com',
        services: ['Website', 'Domain'],
        joinDate: '2024-01-15'
      },
      {
        id: 'CUS-002',
        name: 'Sana Khan',
        company: 'Digital Marketing Pro',
        phone: '+92 301 2345678',
        email: 'sana@dmp.com',
        services: ['Software', 'Hosting'],
        joinDate: '2024-01-20'
      },
      {
        id: 'CUS-003',
        name: 'Bilal Ahmed',
        company: 'Startup Innovations',
        phone: '+92 302 3456789',
        email: 'bilal@startup.com',
        services: ['Website'],
        joinDate: '2024-02-01'
      },
      {
        id: 'CUS-004',
        name: 'Hira Malik',
        company: 'E-commerce Store',
        phone: '+92 303 4567890',
        email: 'hira@estore.com',
        services: ['Domain', 'Hosting', 'Website'],
        joinDate: '2024-02-05'
      },
      {
        id: 'CUS-005',
        name: 'Usman Ali',
        company: 'Consulting Firm',
        phone: '+92 304 5678901',
        email: 'usman@consulting.com',
        services: ['Software'],
        joinDate: '2024-02-10'
      }
    ];

    // ===== Customer Search Functionality =====
    const customerSearch = document.getElementById('customerSearch');
    const searchResults = document.getElementById('searchResults');
    const selectedCustomer = document.getElementById('selectedCustomer');

    customerSearch.addEventListener('input', function(e) {
      const searchTerm = e.target.value.toLowerCase();
      
      if (searchTerm.length < 2) {
        searchResults.style.display = 'none';
        return;
      }
      
      const filteredCustomers = customers.filter(customer => 
        customer.name.toLowerCase().includes(searchTerm) ||
        customer.email.toLowerCase().includes(searchTerm) ||
        customer.phone.includes(searchTerm) ||
        customer.company.toLowerCase().includes(searchTerm)
      );
      
      displaySearchResults(filteredCustomers);
    });

    function displaySearchResults(results) {
      searchResults.innerHTML = '';
      
      if (results.length === 0) {
        searchResults.innerHTML = '<div class="search-result">No customers found</div>';
      } else {
        results.forEach(customer => {
          const resultElement = document.createElement('div');
          resultElement.className = 'search-result';
          resultElement.innerHTML = `
            <div style="font-weight: 600;">${customer.name}</div>
            <div style="font-size: 12px; color: var(--muted);">${customer.email} • ${customer.company}</div>
          `;
          resultElement.addEventListener('click', () => selectCustomer(customer));
          searchResults.appendChild(resultElement);
        });
      }
      
      searchResults.style.display = 'block';
    }

    function selectCustomer(customer) {
      // Update selected customer display
      document.getElementById('customerAvatar').textContent = customer.name.split(' ').map(n => n[0]).join('').toUpperCase();
      document.getElementById('customerName').textContent = customer.name;
      document.getElementById('customerEmail').textContent = customer.email;
      document.getElementById('customerPhone').textContent = customer.phone;
      document.getElementById('customerCompany').textContent = customer.company;
      document.getElementById('customerServices').textContent = customer.services.join(', ');
      
      const joinDate = new Date(customer.joinDate).toLocaleDateString('en-US', {
        year: 'numeric',
        month: 'short',
        day: 'numeric'
      });
      document.getElementById('customerJoinDate').textContent = joinDate;
      
      // Update summary
      document.getElementById('summaryCustomer').textContent = customer.name;
      
      // Show selected customer section
      selectedCustomer.style.display = 'block';
      
      // Hide search results
      searchResults.style.display = 'none';
      customerSearch.value = '';
    }

    // ===== Reason Selection =====
    document.querySelectorAll('.reason-option').forEach(option => {
      option.addEventListener('click', function() {
        document.querySelectorAll('.reason-option').forEach(opt => {
          opt.classList.remove('selected');
        });
        this.classList.add('selected');
        
        // Update summary
        const reasonText = this.querySelector('div > div:first-child').textContent;
        document.getElementById('summaryReason').textContent = reasonText;
      });
    });

    // ===== Date Handling =====
    const removalDate = document.getElementById('removalDate');
    removalDate.addEventListener('change', function() {
      const date = new Date(this.value).toLocaleDateString('en-US', {
        year: 'numeric',
        month: 'short',
        day: 'numeric'
      });
      document.getElementById('summaryDate').textContent = date;
    });

    // ===== Checkbox Updates =====
    document.getElementById('backupData').addEventListener('change', function() {
      document.getElementById('summaryBackup').textContent = this.checked ? 'Enabled' : 'Disabled';
    });

    // ===== Form Validation and Submission =====
    document.getElementById('confirmRemovalBtn').addEventListener('click', function() {
      if (!validateRemovalForm()) return;
      
      if (confirm('Are you sure you want to permanently remove this customer? This action cannot be undone.')) {
        // In a real application, this would submit the removal request
        alert('Customer removal request submitted successfully!');
        window.location.href = 'removallist.html';
      }
    });

    document.getElementById('reviewRemovalBtn').addEventListener('click', function() {
      if (!validateRemovalForm()) return;
      
      // In a real application, this would mark the removal for review
      alert('Customer removal marked for review!');
      window.location.href = 'removallist.html';
    });

    function validateRemovalForm() {
      const customerSelected = selectedCustomer.style.display !== 'none';
      const reasonSelected = document.querySelector('.reason-option.selected');
      const removalDateSet = removalDate.value;
      
      if (!customerSelected) {
        alert('Please select a customer to remove');
        customerSearch.focus();
        return false;
      }
      
      if (!reasonSelected) {
        alert('Please select a removal reason');
        return false;
      }
      
      if (!removalDateSet) {
        alert('Please set a removal date');
        removalDate.focus();
        return false;
      }
      
      return true;
    }

    // ===== Initialize the page =====
    document.addEventListener('DOMContentLoaded', function() {
      // Set current page as active in sidebar
      const currentPage = 'customerremoval.html';
      document.querySelectorAll('.nav a').forEach(link => {
        if (link.getAttribute('href') === currentPage) {
          link.classList.add('active');
        }
      });
      
      // Set default removal date to today
      const today = new Date().toISOString().split('T')[0];
      removalDate.value = today;
      removalDate.dispatchEvent(new Event('change'));
      
      // Close search results when clicking outside
      document.addEventListener('click', function(e) {
        if (!customerSearch.contains(e.target) && !searchResults.contains(e.target)) {
          searchResults.style.display = 'none';
        }
      });
    });
